/*
 * Copyright © 2009 DV Bern AG, Switzerland
 *
 * Das vorliegende Dokument, einschliesslich aller seiner Teile, ist urheberrechtlich
 * geschuetzt. Jede Verwertung ist ohne Zustimmung der DV Bern AG unzulaessig. Dies gilt
 * insbesondere fuer Vervielfaeltigungen, die Einspeicherung und Verarbeitung in
 * elektronischer Form. Wird das Dokument einem Kunden im Rahmen der Projektarbeit zur
 * Ansicht uebergeben ist jede weitere Verteilung durch den Kunden an Dritte untersagt.
 *
 * $Id$
 */
package ch.dvbern.datatypes;

import java.io.Serializable;

/**
 * Basisklasse einer Nummer, welche eine Prüfziffer beinhaltet.
 *
 * @author beph
 */
public abstract class AbstractPruefzifferNummer implements Serializable {

	/**
	 * Serial Version UID
	 */
	private static final long serialVersionUID = 2480682944154690382L;

	private final long nummer;

	private final int pruefziffer;

	/**
	 * @param nummerToCheck
	 * @param minValue
	 * @param maxValue
	 */
	@SuppressWarnings("boxing")
	private void checkStellen(final long nummerToCheck, final long minValue, final long maxValue) {

		if (nummerToCheck < minValue || nummerToCheck > maxValue) {
			throw new IllegalArgumentException("Argument must be between "+minValue+" and "+maxValue);
		}
	}

	/**
	 * Konstruktor einer neuen Sozialversicherungsnummer anhand eines long.
	 *
	 * @param nummer
	 * @param minValue
	 * @param maxValue
	 */
	public AbstractPruefzifferNummer(final long nummer, final long minValue, final long maxValue) {

		checkStellen(nummer, minValue, maxValue);
		this.nummer = nummer;
		this.pruefziffer = berechnePruefziffer(nummer);
	}

	/**
	 * Berechnen der Prüfziffer
	 *
	 * @param nummerToCalculate die Nummer für welche die Prüfziffer erstellt werden muss
	 * @return die berechente Prüfziffer
	 */
	protected abstract int berechnePruefziffer(long nummerToCalculate);

	/**
	 * Konstruktor einer Sozialversicherungsnummer anhand eines Strings.
	 *
	 * @param nummer
	 * @param minValue
	 * @param maxValue
	 */
	public AbstractPruefzifferNummer(final String nummer, final long minValue, final long maxValue) {

		this(Long.parseLong(nummer.replaceAll("[^0-9]", "")), minValue, maxValue);
	}

	/**
	 * @return Returns the nummer.
	 */
	public long getNummer() {

		return nummer;
	}

	/**
	 * @return Returns the pruefziffer.
	 */
	public int getPruefziffer() {

		return pruefziffer;
	}

	/**
	 * Gibt <code>true</code> zurück, wenn die letzte Ziffer der berechneten Prüfziffer entspricht, sonst false.
	 *
	 * @return true wenn
	 * @see #getPruefziffer()
	 */
	public boolean isValid() {

		return nummer % 10 == pruefziffer;
	}

	/**
	 * Gibt die Nummer als String zurück. Convenience- Methode für
	 * <pre>
	 * 	<code>Long.toString(getNumber())</code>
	 * </pre>
	 *
	 * @return die Nummer als String
	 */
	public String getNumberString() {
		return Long.toString(nummer);
	}


    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        AbstractPruefzifferNummer that = (AbstractPruefzifferNummer) o;

        if (nummer != that.nummer) return false;
        return pruefziffer == that.pruefziffer;

    }

    @Override
    public int hashCode() {
        int result = (int) (nummer ^ (nummer >>> 32));
        result = 31 * result + pruefziffer;
        return result;
    }
}
