package ch.dvbern.datatypes;

/*
 * Copyright © 2009 DV Bern AG, Switzerland
 *
 * Das vorliegende Dokument, einschliesslich aller seiner Teile, ist urheberrechtlich
 * geschuetzt. Jede Verwertung ist ohne Zustimmung der DV Bern AG unzulaessig. Dies gilt
 * insbesondere fuer Vervielfaeltigungen, die Einspeicherung und Verarbeitung in
 * elektronischer Form. Wird das Dokument einem Kunden im Rahmen der Projektarbeit zur
 * Ansicht uebergeben ist jede weitere Verteilung durch den Kunden an Dritte untersagt.
 *
 * $Id$
 */

/**
 * Postkontonummer, Modulo 10 rekursiv
 * 
 * @author hefr
 */
public final class PostkontoNummer extends AbstractPruefzifferNummer implements Comparable<PostkontoNummer> {

	private static final long serialVersionUID = 1L;
	private static final long MAX_VALUE = 999999999L;
	private static final long MIN_VALUE = 100000000L;
	private static final int[] TABELLE = { 0, 9, 4, 6, 8, 2, 7, 1, 3, 5 };

	/**
	 * Konstruktor einer neuen Postkontonummer anhand eines long.
	 * 
	 * @param nummer
	 */
	public PostkontoNummer(final long nummer) {

		super(checkDigits(String.valueOf(nummer)), MIN_VALUE, MAX_VALUE);
	}

	/**
	 * Konstruktor einer Postkontonummer anhand eines Strings.
	 * 
	 * @param nummer
	 */
	public PostkontoNummer(final String nummer) {

		super(checkDigits(nummer), MIN_VALUE, MAX_VALUE);
	}

	/**
	 * Postkontonummern können 8 oder 9-stellig sein: XX-XXXXX-X oder XX-XXXXXX-X Im Mittelteil muss also gegebenenfalls
	 * eine 0 ergänzt werden.
	 * 
	 * @param nummer
	 * @return
	 */
	private static String checkDigits(final String nummer) {
        // group1: erste 2 Zeichen
        // group3: letztes Zeichen
        // group2: Rest weniger eventuelle "-" (5 oder 6-stellig)
        String group1 = nummer.substring(0, 2);
        String group3 = nummer.substring(nummer.length()-1);
        String group2 = nummer.substring(2, nummer.length()-1);
        group2 = group2.replaceAll("\\D", "");

        StringBuilder sb = new StringBuilder();
        sb.append(group1);
        // Die mittlere Gruppe muss schlussendlich 6-stellig sein: mit 0 auffüllen
        while (group2.length() <= 5) {
            group2 = "0" + group2;
        }
        sb.append(group2);
        sb.append(group3);

        return sb.toString();
	}

	/**
	 * Berechnet die Pruefziffer.
	 */
	@Override
	protected int berechnePruefziffer(final long nummer) {

		String numberAsString = String.valueOf(nummer / 10);
		int uebertrag = 0;

		byte[] bytes = numberAsString.getBytes();
		for (byte b : bytes) {
			uebertrag = TABELLE[(uebertrag + b - '0') % 10];
		}

		return (10 - uebertrag) % 10;
	}

	@Override
	public String toString() {

		StringBuilder sb = new StringBuilder(Long.toString(getNummer()));
		sb.insert(2, "-");
		sb.insert(9, "-");
		return sb.toString();
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(final PostkontoNummer o) {

		if (this.getNummer() == o.getNummer()) {
			return 0;
		}
		return this.getNummer() - o.getNummer() > 0 ? -1 : 1;
	}

}
