/*
 * Decompiled with CFR 0.152.
 */
package ch.dvbern.lib.update;

import ch.dvbern.lib.update.IOUtils;
import ch.dvbern.lib.update.IProgressListener;
import ch.dvbern.lib.update.OsUtil;
import ch.dvbern.lib.update.UpdateProvider;
import ch.dvbern.lib.update.UpdateVersion;
import ch.dvbern.lib.update.VersionNumber;
import ch.dvbern.lib.update.VersionUtil;
import ch.dvbern.lib.update.exception.CauseType;
import ch.dvbern.lib.update.exception.CommunicationException;
import ch.dvbern.lib.update.exception.ConfigurationException;
import ch.dvbern.lib.update.exception.UpdateFailureException;
import ch.dvbern.lib.update.type.ArchiveItem;
import ch.dvbern.lib.update.type.DownloadableItem;
import ch.dvbern.lib.update.type.FileItem;
import ch.dvbern.lib.update.type.Item;
import ch.dvbern.lib.update.xjc.Delete;
import ch.dvbern.lib.update.xjc.UpdateSite;
import ch.dvbern.lib.update.xjc.Version;
import ch.dvbern.lib.update.xjc.VersionConfig;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.RandomAccessFile;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.TreeSet;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nonnull;
import javax.net.ssl.SSLHandshakeException;

public class UpdateManager {
    private static final Logger LOG = Logger.getLogger(UpdateManager.class.getName());
    public static final String CLEAN_LIST_FNAME = "clean-list.txt";
    public static final Charset UTF_8 = Charset.forName("UTF-8");
    private static final Pattern PATH_PTN = Pattern.compile("(/*)(.*?)(/*)");
    @Nonnull
    private final UpdateProvider updateProvider;
    private final NavigableSet<UpdateVersion> updateVersions;
    private final VersionConfig versionConfig;
    private final VersionNumber localVersion;
    @Nonnull
    private final List<IProgressListener> progressListeners = new CopyOnWriteArrayList<IProgressListener>();

    private UpdateManager(@Nonnull UpdateProvider updateProvider, NavigableSet<UpdateVersion> updateVersions, VersionConfig versionConfig, VersionNumber localVersion) {
        this.updateProvider = Objects.requireNonNull(updateProvider, "update provider cannot be null");
        this.updateVersions = updateVersions;
        this.versionConfig = versionConfig;
        this.localVersion = localVersion;
    }

    public static UpdateManager openUpdateSite(@Nonnull UpdateProvider updateProvider) throws UpdateFailureException, CommunicationException, ConfigurationException {
        UpdateSite updateSite;
        Objects.requireNonNull(updateProvider);
        VersionConfig versionConfig = VersionUtil.loadVersionConfig();
        VersionNumber localVersion = VersionNumber.parse(versionConfig.getVersion());
        try (InputStream is = updateProvider.getUpdateSite();){
            updateSite = VersionUtil.loadUpdateSite(is);
        }
        catch (ConfigurationException e) {
            throw new UpdateFailureException(CauseType.SERVER, "Failed to parse update site configuration", e);
        }
        catch (SSLHandshakeException ssl) {
            throw new UpdateFailureException(CauseType.SERVER, "Could not establish SSL connection. Remote certificate invalid?", ssl);
        }
        catch (IOException e) {
            throw new CommunicationException(e.getMessage(), e);
        }
        String localeAppName = versionConfig.getAppName();
        String updateSiteAppName = updateSite.getAppName();
        if (!localeAppName.equals(updateSiteAppName)) {
            throw new UpdateFailureException(CauseType.SERVER, "Mismatch between locale application name (" + localeAppName + ") and update site application name (" + updateSiteAppName + ")");
        }
        List<Version> versions = updateSite.getVersions();
        TreeSet<UpdateVersion> updateVersions = new TreeSet<UpdateVersion>();
        for (Version version : versions) {
            if (updateVersions.add(UpdateVersion.load(version))) continue;
            throw new UpdateFailureException(CauseType.SERVER, "Duplicate version number in update-site: " + version.getNumber());
        }
        return new UpdateManager(updateProvider, Collections.unmodifiableNavigableSet(updateVersions), versionConfig, localVersion);
    }

    public boolean isUpdateAvailable() {
        UpdateVersion lastVersion = (UpdateVersion)this.updateVersions.last();
        if (lastVersion == null) {
            LOG.info("No version defined in update-site.");
            return false;
        }
        int compare = this.localVersion.compareTo(lastVersion.getVersionNumber());
        if (compare < 0) {
            LOG.info("An update is available, current: " + this.localVersion + ", target: " + lastVersion);
            return true;
        }
        if (compare == 0) {
            LOG.info("No update to apply: already at the latest version");
            return false;
        }
        assert (compare > 0);
        LOG.warning("Locale version " + this.localVersion + " greater than latest update " + lastVersion);
        return false;
    }

    public NavigableSet<UpdateVersion> getAvailableVersions() {
        return this.updateVersions;
    }

    public NavigableSet<UpdateVersion> getAvailableUpdates() {
        return this.updateVersions.tailSet(new UpdateVersion(this.localVersion, null), false);
    }

    public void update() throws CommunicationException, UpdateFailureException, InterruptedException {
        if (this.updateVersions.isEmpty()) {
            throw new IllegalStateException("No version to be updated.");
        }
        this.update((UpdateVersion)this.updateVersions.last());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void update(UpdateVersion targetVersion) throws UpdateFailureException, CommunicationException, InterruptedException {
        Objects.requireNonNull(targetVersion, "Update version cannot be null.");
        if (!this.updateVersions.contains(targetVersion)) {
            throw new IllegalArgumentException("Unknown update version: " + targetVersion);
        }
        if (targetVersion.getVersionNumber().compareTo(this.localVersion) <= 0) {
            throw new IllegalArgumentException("Cannot upgrade from " + this.localVersion + " to " + targetVersion + ".");
        }
        if (this.isUpdateAvailable()) {
            File lockFile = OsUtil.getUpdateLockFile(this.versionConfig);
            lockFile.getParentFile().mkdirs();
            if (!lockFile.getParentFile().exists()) {
                throw new IllegalStateException("Could not init lockfile durectory: " + lockFile);
            }
            RandomAccessFile updateLock = null;
            try {
                updateLock = OsUtil.tryLockSilently(lockFile);
                if (updateLock != null) {
                    this.performUpdate(targetVersion);
                } else {
                    LOG.log(Level.WARNING, "Cannot acquire update lock, aborting update");
                }
            }
            catch (IOException e) {
                LOG.log(Level.SEVERE, "I/O error while acquiring the update lock", e);
            }
            finally {
                IOUtils.closeQuietly(updateLock);
            }
        }
    }

    /*
     * WARNING - void declaration
     */
    private void performUpdate(UpdateVersion targetVersion) throws UpdateFailureException, CommunicationException, InterruptedException {
        boolean result;
        assert (this.versionConfig != null && this.localVersion.compareTo(targetVersion.getVersionNumber()) < 0);
        File inProgressDir = OsUtil.getDownloadInProgressDir(this.versionConfig);
        File completeDir = OsUtil.getDownloadCompletedDir(this.versionConfig);
        IOUtils.deleteFileOrDirectory(completeDir);
        inProgressDir.mkdirs();
        if (!inProgressDir.exists()) {
            throw new IllegalStateException("Could not init inProgressDir: " + inProgressDir);
        }
        LinkedHashSet<String> cleanList = new LinkedHashSet<String>();
        LinkedHashMap<String, DownloadItem> downloadItemMap = new LinkedHashMap<String, DownloadItem>();
        NavigableSet<UpdateVersion> toUpdate = this.updateVersions.subSet(new UpdateVersion(this.localVersion, null), false, targetVersion, true);
        for (UpdateVersion version : toUpdate) {
            Objects.requireNonNull(version.getXmlDefinition(), "UpdateVersion incorrectly initialized with xmlDefinition=null");
            for (Object obj : version.getXmlDefinition().getFilesAndArchivesAndDeletes()) {
                if (!(obj instanceof Item)) {
                    throw new UpdateFailureException(CauseType.SERVER, "Unknown version item type, check generated API: " + obj.getClass().getName());
                }
                Item item = (Item)obj;
                if (!OsUtil.correctOperatingSystem(item.getOs()) || !OsUtil.correctOsArch(item.getArch())) continue;
                if (item instanceof Delete) {
                    cleanList.add(UpdateManager.buildPath(item.getTargetLocation(), item.getTargetName()));
                    continue;
                }
                if (item instanceof DownloadableItem) {
                    String path;
                    DownloadableItem downloadableItem = (DownloadableItem)item;
                    String string = downloadableItem.getTargetLocation();
                    String targetName = downloadableItem.getTargetName();
                    boolean blankTargetLocation = VersionUtil.isBlank(string);
                    boolean blankTargetName = VersionUtil.isBlank(targetName);
                    if (item instanceof FileItem) {
                        void var13_19;
                        FileItem file = (FileItem)item;
                        if (blankTargetLocation) {
                            String string2 = file.getLocation();
                        }
                        targetName = file.getTargetName();
                        if (blankTargetName) {
                            targetName = file.getName();
                        }
                        path = UpdateManager.buildPath((String)var13_19, targetName);
                    } else if (item instanceof ArchiveItem) {
                        void var13_21;
                        if (blankTargetLocation) {
                            String string3 = "";
                        }
                        if (blankTargetName) {
                            throw new UpdateFailureException(CauseType.SERVER, "Target name is mandatory for archives item, check XML schema");
                        }
                        path = UpdateManager.buildPath((String)var13_21, targetName);
                        cleanList.add(path);
                    } else {
                        throw new UpdateFailureException(CauseType.SERVER, "Unknown downloadable item type: " + item.getClass().getName());
                    }
                    downloadItemMap.put(path, new DownloadItem(version, downloadableItem));
                    continue;
                }
                throw new UpdateFailureException(CauseType.SERVER, "Unknown version item type, check generated API: " + obj.getClass().getName());
            }
        }
        if (downloadItemMap.isEmpty()) {
            throw new UpdateFailureException(CauseType.SERVER, "At least an update of the version configuration file must be resolved for each update.");
        }
        long totalSize = 0L;
        long downloadedSize = 0L;
        Collection downloadItems = downloadItemMap.values();
        for (DownloadItem downloadItem : downloadItems) {
            totalSize += downloadItem.item.getSize();
        }
        for (Map.Entry entry : downloadItemMap.entrySet()) {
            this.checkInterrupt();
            DownloadItem downloadItem = (DownloadItem)entry.getValue();
            try {
                this.download(inProgressDir, (String)entry.getKey(), downloadItem, downloadedSize, totalSize);
            }
            catch (IOException e) {
                throw new CommunicationException("File download failed", e);
            }
            this.updateProgress(Math.round(100.0f * (float)(downloadedSize += downloadItem.item.getSize()) / (float)totalSize));
        }
        if (!cleanList.isEmpty()) {
            PrintWriter pw = null;
            try {
                pw = new PrintWriter(new File(inProgressDir, CLEAN_LIST_FNAME), "UTF-8");
                for (String toBeCleaned : cleanList) {
                    pw.println(toBeCleaned);
                }
            }
            catch (IOException iOException) {
                try {
                    throw new UpdateFailureException(CauseType.SERVER, "Failed to create clean-up list", iOException);
                }
                catch (Throwable throwable) {
                    IOUtils.closeQuietly(pw);
                    throw throwable;
                }
            }
            IOUtils.closeQuietly(pw);
        }
        this.checkInterrupt();
        if (completeDir.exists()) {
            IOUtils.deleteFileOrDirectory(completeDir);
        }
        boolean bl = result = !inProgressDir.exists() || inProgressDir.renameTo(completeDir);
        if (result) {
            this.updateFinished();
        }
    }

    private void checkInterrupt() throws InterruptedException {
        if (Thread.currentThread().isInterrupted()) {
            this.updateCancelled();
            throw new InterruptedException();
        }
    }

    static String buildPath(String location, String name) throws UpdateFailureException {
        if (location == null || name == null) {
            throw new NullPointerException("invalid null argument");
        }
        StringBuilder sb = new StringBuilder();
        Matcher locationMatcher = PATH_PTN.matcher(location);
        if (!locationMatcher.matches()) {
            assert (false) : "Location pattern should accept any string";
            throw new UpdateFailureException(CauseType.SERVER, "Failed to parse location");
        }
        sb.append(locationMatcher.group(2));
        Matcher nameMatcher = PATH_PTN.matcher(name);
        if (!nameMatcher.matches()) {
            assert (false) : "Location pattern should accept any string";
            throw new UpdateFailureException(CauseType.SERVER, "Failed to parse location");
        }
        sb.append('/');
        sb.append(nameMatcher.group(2));
        return sb.toString();
    }

    public void addProgressListener(IProgressListener l) {
        if (l == null) {
            throw new NullPointerException("listener cannot be null");
        }
        this.progressListeners.add(l);
    }

    public void removeProgressListener(IProgressListener l) {
        if (l == null) {
            throw new NullPointerException("listener cannot be null");
        }
        this.progressListeners.remove(l);
    }

    private void updateProgress(String resource) {
        for (IProgressListener listener : this.progressListeners) {
            listener.updateProgress(resource);
        }
    }

    private void updateProgress(int percent) {
        for (IProgressListener listener : this.progressListeners) {
            listener.updateProgress(percent);
        }
    }

    private void updateFinished() {
        for (IProgressListener progressListener : this.progressListeners) {
            progressListener.updateFinished();
        }
    }

    private void updateCancelled() {
        for (IProgressListener progressListener : this.progressListeners) {
            progressListener.updateCancelled();
        }
    }

    private void download(@Nonnull File targetDir, @Nonnull String target, @Nonnull DownloadItem downloadItem, long downloadedSize, long totalSize) throws IOException, UpdateFailureException, InterruptedException, FileNotFoundException {
        Objects.requireNonNull(targetDir);
        Objects.requireNonNull(target);
        Objects.requireNonNull(downloadItem);
        String path = downloadItem.version.getVersionNumber().asText() + "/" + UpdateManager.buildPath(downloadItem.item.getLocation(), downloadItem.item.getName());
        File targetFile = new File(targetDir, target);
        this.ensureParentDirectories(targetFile);
        File download = downloadItem.isArchive() ? File.createTempFile("dvb_lib_upd", ".tmp") : targetFile;
        this.updateProgress(downloadItem.item.getDesc());
        this.downloadFileWithProgress(download, path, downloadedSize, totalSize);
        URL fileURL = this.updateProvider.buildURL(path);
        this.updateProvider.getSignatureVerifier().verify(fileURL, Files.readAllBytes(download.toPath()), this.updateProvider);
        if (downloadItem.isArchive()) {
            try {
                IOUtils.unzip(download, target, targetDir);
            }
            catch (NoSuchElementException e) {
                throw new UpdateFailureException(CauseType.SERVER, "Cannot found entry " + target + " in archive " + path, e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void downloadFileWithProgress(File localFile, String path, long downloadedSize, long totalSize) throws IOException, InterruptedException, FileNotFoundException {
        InputStream is = null;
        FileOutputStream os = null;
        try {
            int bytesRead;
            is = this.updateProvider.getFile(path);
            os = new FileOutputStream(localFile);
            byte[] buffer = new byte[8196];
            int totalBytesRead = 0;
            while ((bytesRead = is.read(buffer)) != -1) {
                this.checkInterrupt();
                os.write(buffer, 0, bytesRead);
                this.updateProgress(Math.round(100.0f * (float)((long)(totalBytesRead += bytesRead) + downloadedSize) / (float)totalSize));
            }
        }
        catch (Throwable throwable) {
            IOUtils.closeQuietly(is);
            IOUtils.closeQuietly(os);
            throw throwable;
        }
        IOUtils.closeQuietly(is);
        IOUtils.closeQuietly(os);
    }

    private void ensureParentDirectories(File targetFile) {
        File targetFileParent = targetFile.getParentFile();
        if (targetFileParent != null) {
            targetFileParent.mkdirs();
            if (!targetFileParent.exists()) {
                throw new IllegalStateException("Could not init parent of target: " + targetFile);
            }
        }
    }

    private static class DownloadItem {
        private final UpdateVersion version;
        private final DownloadableItem item;

        public DownloadItem(UpdateVersion version, DownloadableItem item) {
            assert (version != null && item != null);
            this.version = version;
            this.item = item;
        }

        public boolean isArchive() {
            return this.item instanceof ArchiveItem;
        }
    }
}

